//
//  MicroLifeCloudManager.h
//  MicroLifeDeviceSDK
//
//  Created by 吳憲有 on 2018/10/23.
//  Copyright © 2018 WiFi@MBP. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <UIKit/UIKit.h>
#import <SafariServices/SafariServices.h>

NS_ASSUME_NONNULL_BEGIN
typedef NS_ENUM(NSInteger, GrantType) {
    authorization_code = 0,
    refresh_token,
};

/**
 主機使用

 - MicroLifeDevelopSever: 開發主機
 - MicroLifeTestSever: 測試主機
 - MicroLifeProductSever: 正式主機
 */
typedef NS_ENUM(NSInteger, MicroLifeSeverType) {
    MicroLifeDevelopSever = 0,
    MicroLifeTestSever,
    MicroLifeProductSever,
};

@interface MicroLifeCloudManager : NSObject

@property (nonatomic, assign) NSInteger code;
@property (nonatomic, copy)   NSString * info;
@property (nonatomic, strong) id  data;
@property (nonatomic, copy)   NSString * access_token;
@property (nonatomic, copy)   NSString * refresh_token;
@property (nonatomic, copy)   NSString * api_key;
@property (nonatomic, assign) NSInteger sys_id;
@property (nonatomic, assign) NSInteger account_id;
@property (nonatomic, copy)   NSString * photo;
@property (nonatomic, copy)   NSString * recording;
@property (nonatomic, copy)   NSString * recording_time;
@property (nonatomic, assign) NSInteger total_num;
@property (nonatomic, assign) NSInteger mail_id;
@property (nonatomic, strong) NSArray * bpm;
@property (nonatomic, strong) NSArray * weight;
@property (nonatomic, strong) NSArray * bt;
@property (nonatomic, copy)   NSString * pdf_path;

#pragma mark 設定使用主機
+ (void)setMainSeverType:(MicroLifeSeverType)severType;

/**
 login

 @param vc UIViewController(self)
 @param client_id client_id
 @param redirect_uri redirect_uri
 @param lang Null value is the system language,The language code is as follows:'tw', 'english', 'french', 'greman', 'italian', 'polish', 'russian', 'spanish', "cn", "czech", "slovak"
 @param flag animated
 @param completion The completion handler.
 */
+ (void)loginWithVC:(id)vc ClientId:(NSString *)client_id RedirectURL:(NSString *)redirect_uri Lang:(id)lang animated:(BOOL)flag completion:(void (^)(void))completion;

/**
 Change Password

 @param vc UIViewController(self)
 @param lang Null value is the system language,The language code is as follows:'tw', 'english', 'french', 'greman', 'italian', 'polish', 'russian', 'spanish', "cn", "czech", "slovak"
 @param flag animated
 @param completion The completion handler.
 */
+ (void)changePasswordWithVC:(id)vc Lang:(id)lang animated:(BOOL)flag completion:(void (^)(void))completion;

/**
 Obtain access_token

 @param grant_type Request method
 @param code code
 @param refresh_token refresh_token
 @param client_id Developer client_id
 @param client_secret Developer client_secret
 @param redirect_uri url scheme
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postOAuthToken:(GrantType)grant_type code:(NSString *)code refreshToken:(NSString *)refresh_token clientID:(NSString *)client_id clientSecret:(NSString *)client_secret redirectURI:(NSString *)redirect_uri FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Mobile phone information recorded by members

 @param token oauth2_token
 @param client_unique_id Device unique value
 @param os Equipment system
 @param machine_type Equipment type
 @param push_unique_id Push-only code
 @param model Equipment model
 @param company Equipment company
 @param gps gps
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postSys:(NSString *)token client_unique_id:(NSString *)client_unique_id os:(NSString *)os machine_type:(NSString *)machine_type push_unique_id:(NSString *)push_unique_id model:(NSString *)model company:(NSString *)company gps:(NSString *)gps FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Forgot password [disabled]

 @param account account number
 @param client_id Developer client_id
 @param client_secret Developer client_secret
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postForgetPassword:(NSString *)account client_id:(NSString *)client_id client_secret:(NSString *)client_secret FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Added member operation record

 @param access_token oauth2_token
 @param actionArray  "{\"data\":[{  \"date\" : \"2019-03-29 15:25:42\",  \"log_action\" : \"16\",  \"device_type\" : \"0\"}]}";
 device_type [0] no equipment [1] blood pressure machine, [2] weight machine, [3] front temperature gun (device of this data)
 log_action Member opcode, pure number, not yet defined.  Reference link:
 https://docs.google.com/spreadsheets/d/1HlZV850T7q-T8lqjnRMYaNFUW_uEJUx7q4mlTAl3gWM/edit#gid=0
 date Operation date
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postAddMemberActionLog:(NSString *)access_token ActionArray:(NSMutableArray *)actionArray FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Get basic member information
 
 @param access_token oauth2_token
 @param client_id Developer client_id
 @param client_secret Developer client_secret
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postGetMemberBaseData:(NSString *)access_token client_id:(NSString *)client_id client_secret:(NSString *)client_secret FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Get member information

 @param access_token oauth2_token
 @param client_id Developer client_id
 @param client_secret Developer client_secret
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postGetMemberData:(NSString *)access_token client_id:(NSString *)client_id client_secret:(NSString *)client_secret FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Modify personal information

 @param access_token oauth2_token
 @param client_id Developer client_id
 @param client_secret Developer client_secret
 @param name Name
 @param birthday birthday
 @param gender Gender 0 : Male, 1: Female
 @param height height
 @param weight body weight
 @param unit_type Unit category 0: Metric, 1: Imperial
 @param sys_unit Target systolic pressure unit 0: mmHg, 1: kpa
 @param sys Target systolic pressure
 @param dia Target diastolic pressure
 @param goal_weight Target weight
 @param body_fat Target body fat
 @param bmi Target BMI
 @param sys_activity Whether to turn on the target systolic pressure 0 (preset value), 0=OFF, 1=ON
 @param dia_activity Whether to open the target diastolic pressure 0 (preset value), 0=OFF, 1=ON
 @param weight_activity Whether to turn on the target weight 0 (default), 0=OFF, 1=ON
 @param body_fat_activity Whether to open the target body fat 0 (default value), 0=OFF, 1=ON
 @param bmi_activity Whether to turn on the target BMI 0 (default), 0=OFF, 1=ON
 @param threshold Normal value 0 (default), 0=OFF, 1=ON
 @param cuff_size Armband size 0 (default)
 @param bp_measurement_arm Measuring arm 0 (default)
 @param date_format Date format 1 (default) 0=2018/01/01, 1=01.01.2018
 @param conditions disease
 @param photo Big head photo
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postModifyMember:(NSString *)access_token client_id:(NSString *)client_id client_secret:(NSString *)client_secret name:(NSString *)name birthday:(NSString *)birthday gender:(int)gender height:(int)height weight:(float)weight unit_type:(int)unit_type sys_unit:(int)sys_unit sys:(float)sys dia:(float)dia goal_weight:(float)goal_weight body_fat:(float)body_fat bmi:(float)bmi sys_activity:(int)sys_activity dia_activity:(int)dia_activity weight_activity:(int)weight_activity body_fat_activity:(int)body_fat_activity bmi_activity:(int)bmi_activity threshold:(int)threshold cuff_size:(int)cuff_size bp_measurement_arm:(int)bp_measurement_arm date_format:(int)date_format conditions:(NSString *)conditions photo:(UIImage *)photo FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Blood pressure machine history data

 @param access_token oauth2_token
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postGetBPMHistoryData:(NSString *)access_token FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Add blood pressure machine record

 @param access_token oauth2_token
 @param user_id Blood pressure machine ID [9 digits + 2 English EX: 123456789AB]
 @param sys Systolic blood pressure
 @param dia Diastolic blood pressure
 @param pul Heart rate
 @param cuffokr Arm band tightness 0: loose 1: tight 2: no
 @param bpm_id Blood pressure machine code
 @param afib 0: No measurement, 1: Measured
 @param pad 0: No measurement, 1: Measured
 @param man 0: Normal Mode,1: Normal Mode + AFIB Screening,2: MAM Mode,3: MAM Mode + AFIB Screening
 @param date Date [2016-08-03 15:20:30]
 @param mac_address mac_address Manually added: 00:00:00:00:00:00
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postAddBPM:(NSString *)access_token user_id:(NSString *)user_id sys:(float)sys dia:(float)dia pul:(int)pul cuffokr:(int)cuffokr bpm_id:(NSString *)bpm_id afib:(int)afib pad:(int)pad man:(int)man date:(NSString *)date mac_address:(NSString *)mac_address FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Modify blood pressure machine record

 @param access_token oauth2_token
 @param bpm_id Blood pressure machine code
 @param sys Systolic blood pressure
 @param dia Diastolic blood pressure
 @param pul Heart rate
 @param date Date [2016-08-03 15:20:30]
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postEditBPM:(NSString *)access_token bpm_id:(NSString *)bpm_id sys:(float)sys dia:(float)dia pul:(int)pul date:(NSString *)date FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Weight machine history data

 @param access_token oauth2_token
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postGetWeightHistoryData:(NSString *)access_token FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 New weight record

 @param access_token oauth2_token
 @param weight_id Weight code
 @param weight body weight
 @param bmi BMI
 @param body_fat Body fat
 @param water Body water
 @param skeleton Bone quality
 @param muscle muscle
 @param bmr Basal metabolic rate
 @param organ_fat Visceral fat
 @param date Date [2016-08-03 15:20:30]
 @param mac_address mac_address Manually added: 00:00:00:00:00:00
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postAddWeightData:(NSString *)access_token weight_id:(NSString *)weight_id weight:(float)weight bmi:(float)bmi body_fat:(float)body_fat water:(int)water skeleton:(float)skeleton muscle:(int)muscle bmr:(int)bmr organ_fat:(float)organ_fat date:(NSString *)date mac_address:(NSString *)mac_address FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Change weight record

 @param access_token oauth2_token
 @param weight_id Weight code
 @param weight body weight
 @param bmi BMI
 @param body_fat Body fat
 @param date Date [2016-08-03 15:20:30]
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postEditWeightData:(NSString *)access_token weight_id:(NSString *)weight_id weight:(float)weight bmi:(float)bmi body_fat:(float)body_fat date:(NSString *)date FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Get event list

 @param access_token oauth2_token
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postGetBTEventList:(NSString *)access_token FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 New event

 @param access_token oauth2_token
 @param event_code Event code [not 0]
 @param event Measuring event name
 @param type category
 @param event_time Event time [2018-01-01]
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postAddBTEvent:(NSString *)access_token event_code:(int)event_code event:(NSString *)event type:(NSString *)type event_time:(NSString *)event_time FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Modify or delete an event

 @param access_token oauth2_token
 @param event_cod Event code
 @param event Measuring event name
 @param type category
 @param event_time Event time [2018-01-01]
 @param isDelete delete
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postEditBTEvent:(NSString *)access_token event_code:(int)event_cod event:(NSString *)event type:(NSString *)type event_time:(NSString *)event_time isDelete:(BOOL)isDelete FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Obtain historical data of temperature

 @param access_token oauth2_token
 @param event_code Event code
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postGetBTHistoryData:(NSString *)access_token event_code:(int)event_code FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 New temperature

 @param access_token oauth2_token
 @param event_code Event code
 @param bt_id Temperature code
 @param body_temp body temperature
 @param room_temp room temperature
 @param date Date [2016-08-03 15:20:30]
 @param mac_address mac_address
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postAddBTData:(NSString *)access_token event_code:(int)event_code bt_id:(NSString *)bt_id body_temp:(float)body_temp room_temp:(float)room_temp date:(NSString *)date mac_address:(NSString *)mac_address FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Modify the temperature

 @param access_token oauth2_token
 @param event_code Event code
 @param bt_id Temperature code
 @param body_temp body temperature
 @param room_temp room temperature
 @param date Date [2016-08-03 15:20:30]
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postEditBTData:(NSString *)access_token event_code:(int)event_code bt_id:(NSString *)bt_id body_temp:(float)body_temp room_temp:(float)room_temp date:(NSString *)date FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Add historical data note content

 @param access_token oauth2_token
 @param type_id Blood pressure machine (bpm_id), weight machine (weight_id), front temperature gun (bt_id) (id of three data sheets, according to note_type)
 @param note_type [1] blood pressure machine, [2] weight machine, [3] fore temperature gun (device for this data)
 @param note Notes content
 @param photo photo
 @param recording recording
 @param recording_time Recording time [10:10]
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postAddNoteData:(NSString *)access_token type_id:(NSString *)type_id note_type:(int)note_type note:(NSString *)note photo:(UIImage *)photo recording:(NSData *)recording recording_time:(NSString *)recording_time FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Delete historical data note content

 @param access_token oauth2_token
 @param type_id Blood pressure machine (bpm_id), weight machine (weight_id), front temperature gun (bt_id) (id of three data sheets, according to note_type)
 @param note_type [1] blood pressure machine, [2] weight machine, [3] fore temperature gun (device for this data)
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postDeleteRecordData:(NSString *)access_token type_id:(NSString *)type_id note_type:(NSString *)note_type FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Get mailer list

 @param access_token oauth2_token
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postGetMailNotificationList:(NSString *)access_token FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Add mailer list

 @param access_token oauth2_token
 @param name Name
 @param email mailbox
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postAddMailNotification:(NSString *)access_token name:(NSString *)name email:(NSString *)email FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Modify (delete) the mailer list

 @param access_token oauth2_token
 @param mail_id Judging data
 @param name Name
 @param email mailbox
 @param isDelete delete
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postEditMailNotification:(NSString *)access_token mail_id:(NSMutableArray<NSNumber *> *)mail_id name:(NSString *)name email:(NSString *)email isDelete:(BOOL)isDelete FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Get device list

 @param access_token oauth2_token
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postGetDeviceList:(NSString *)access_token FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 New equipment record

 @param access_token oauth2_token
 @param device_type Device type [1] blood pressure machine, [2] weight machine, [3] fore temperature gun (device for this data)
 @param device_model Device Model
 @param error_code error code
 @param mac_address mac_address
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postAddDevice:(NSString *)access_token device_type:(int)device_type device_model:(NSString *)device_model error_code:(NSString *)error_code mac_address:(NSString *)mac_address FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Delete device data

 @param access_token oauth2_token
 @param mac_address mac_address
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postDeleteDeviceData:(NSString *)access_token mac_address:(NSString *)mac_address FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Delete user

 @param access_token oauth2_token
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postDeletetModifyMember:(NSString *)access_token FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Export blood pressure machine PDF

 @param access_token oauth2_token
 @param lang Null value is the system language,The language code is as follows:'tw', 'english', 'french', 'greman', 'italian', 'polish', 'russian', 'spanish', "cn", "czech", "slovak"
 @param start_date Start time [2016-08-03 15:20:30]
 @param end_date End time [2016-08-03 15:20:30]
 @param sys_threshold Systolic pressure threshold
 @param dia_threshold diastolic pressure threshold
 @param photo Uploaded image file [app on chart]
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postDownloadBPMPDF:(NSString *)access_token lang:(NSString *)lang start_date:(NSString *)start_date end_date:(NSString *)end_date sys_threshold:(int)sys_threshold dia_threshold:(int)dia_threshold photo:(UIImage *)photo FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Remittance temperature gun PDF

 @param access_token oauth2_token
 @param lang Null value is the system language,The language code is as follows:'tw', 'english', 'french', 'greman', 'italian', 'polish', 'russian', 'spanish', "cn", "czech", "slovak"
 @param start_date Start time [2016-08-03 15:20:30]
 @param end_date End time [2016-08-03 15:20:30]
 @param event_code Body temperature event 0~5,0 is all events.
 @param threshold Body temperature threshold
 @param photo Uploaded image file [app on chart]
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postDownloadBTPDF:(NSString *)access_token lang:(NSString *)lang start_date:(NSString *)start_date end_date:(NSString *)end_date eventCode:(int)event_code threshold:(float)threshold photo:(UIImage *)photo FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Export body fat machine PDF

 @param access_token oauth2_token
 @param lang Null value is the system language,The language code is as follows:'tw', 'english', 'french', 'greman', 'italian', 'polish', 'russian', 'spanish', "cn", "czech", "slovak"
 @param start_date Start time [2016-08-03 15:20:30]
 @param end_date End time [2016-08-03 15:20:30]
 @param wei_threshold Weight threshold 0 does not display threshold
 @param bmi_threshold BMI threshold 0 does not display threshold
 @param fat_threshold Body fat threshold 0 does not display threshold
 @param photo Uploaded image file [app on chart]
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postDownloadWeightPDF:(NSString *)access_token lang:(NSString *)lang start_date:(NSString *)start_date end_date:(NSString *)end_date wei_threshold:(float)wei_threshold bmi_threshold:(float)bmi_threshold fat_threshold:(float)fat_threshold photo:(UIImage *)photo FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;

/**
 Get a list of push history

 @param access_token oauth2_token
 @param block bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager
 */
+ (void)postGetPushHistoryList:(NSString *)access_token FinishBlock:(void (^)(bool success, NSInteger code, MicroLifeCloudManager *microLifeCloudManager))block;
@end

NS_ASSUME_NONNULL_END
