//
//  WBPBLEManager.h
//  MicroLifeDeviceSDK-Dome
//
//  Created by willy.wu on 2019/4/12.
//  Copyright © 2019 WiFi@MBP. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "MicroLifeDeviceSDK.h"
#import "MicroLifeDevice.h"
#import "MicroLifeUserInfo.h"
#import "MicroLifeDeviceInfo.h"
#import "MicroLifeDRecord.h"
#import "MicroLifeCurrentAndMData.h"
#import "MicroLifeDiagnosticDRecord.h"
#import "MicroLifeNocturnalModeDRecord.h"

NS_ASSUME_NONNULL_BEGIN
@protocol WBPDataResponseDelegate
#pragma mark - Bluetooth connection section
/**
 Mobile phone Bluetooth status changes
 
 @param state Mobile phone Bluetooth status
 */
- (void)WBPBLEManagerCellPhoneBluetoothDidUpdateState:(MicroLifeBLEState)state;

/**
 Search to device
 
 @param macAddress macAddress
 @param name name
 @param RSSI RSSI
 */
- (void)WBPBLEManagerDidDiscoverBluetoothDeviceMacAddress:(NSData *)macAddress Name:(NSString *)name RSSI:(NSNumber *)RSSI;

/**
 Device is connected
 */
- (void)WBPBLEManagerDidConnectDevice;

/**
 Device connection failed
 */
- (void)WBPBLEManagerDidFailToConnectDevice;

/**
 Disconnect
 */
- (void)WBPBLEManagerDidDisconnectDevice;

#pragma mark - Received data section
/**
 Read usual mode history data from BPM
 
 @param data usual mode history data
 @param isNoData is No Data
 */
- (void)WBPBLEManagerResponseReadUsualModeHistoryData:(MicroLifeDRecord *)data IsNoData:(BOOL)isNoData;

/**
 Read diagnostic mode history data from BPM
 
 @param data diagnostic mode history data
 @param isNoData is No Data
 */
- (void)WBPBLEManagerResponseReadDiagnosticModeHistoryData:(MicroLifeDiagnosticDRecord *)data IsNoData:(BOOL)isNoData;

/**
 Read diagnostic mode history data from BPM
 
 @param isSuccess Success
 */
- (void)WBPBLEManagerResponseClearSelectedModeHistoryData:(BOOL)isSuccess;

/**
 Clear current mode history data of the BPM

 @param isSuccess Success
 */
- (void)WBPBLEManagerResponseClearCurrentModeHistoryData:(BOOL)isSuccess;

/**
 Write device Time to BPM
 
 @param isSuccess Success
 */
- (void)WBPBLEManagerResponseWriteDeviceTime:(BOOL)isSuccess;

/**
 Write a new user ID to BPM
 
 @param isSuccess Success
 */
- (void)WBPBLEManagerResponseWriteUserID:(BOOL)isSuccess;

/**
 Change nocturnal mode setting
 
 @param isSuccess Success
 */
- (void)WBPBLEManagerResponseChangeNocturnalModeSetting:(BOOL)isSuccess;

/**
 Read nocturnal mode setting
 
 @param deviceInfo nocturnal mode setting
 */
- (void)WBPBLEManagerResponseReadNocturnalModeSetting:(MicroLifeDeviceInfo *)deviceInfo;

/**
 Read device ID and info from BPM.
 
 @param deviceInfo device ID and info
 */
- (void)WBPBLEManagerResponseReadDeviceIDAndInfo:(MicroLifeDeviceInfo *)deviceInfo;

/**
 Read device Time from BPM

 @param deviceInfo device Time
 */
- (void)WBPBLEManagerResponseReadDeviceTime:(MicroLifeDeviceInfo *)deviceInfo;

/**
 Read user ID and version data from BPM
 
 @param user user ID
 @param verData version data
 */
- (void)WBPBLEManagerResponseReadUserAndVersionData:(MicroLifeUserInfo *)user VersionData:(MicroLifeDeviceInfo *)verData;

/**
 Read Nocturnal pattern history data from BPM

 @param data Nocturnal pattern history data
 @param isNoData is No Data
 */
- (void)WBPBLEManagerResponseReadNocturnalPatternHistoryData:(MicroLifeNocturnalModeDRecord *)data IsNoData:(BOOL)isNoData;
@end

@interface WBPBLEManager : NSObject
@property (nonatomic, weak) id<WBPDataResponseDelegate> dataResponseDelegate;
@property (nonatomic, strong) MicroLifeDevice *bindingDevice;

@property (nonatomic, strong) NSString *result;
@property (nonatomic, strong) NSString *message;

/**
 Singleton Construction Bluetooth Controller Open equipment:
 "WatchBP Home"
 @param key Authorization key
 @return Bluetooth Controller
 */
+ (instancetype)shareInstanceWhithAuthorizationkey:(NSString *)key;

/**
 Singleton Construction Bluetooth Controller
 
 @param key Authorization key
 @param targetWBPNames Set the specified display Bluetooth device name
 @return Bluetooth Controller
 */
+ (instancetype)shareInstanceWhithAuthorizationkey:(NSString *)key TargetWBPNames:(NSArray *)targetWBPNames;

#pragma mark - Send
/**
 Read usual mode history data from BPM
 */
- (void)readUsualModeHistoryData;

/**
 Read diagnostic mode history data from BPM
 */
- (void)readDiagnosticModeHistoryData;

/**
 Clear selected mode history data of the BPM

 @param clearUsualMode clear Usual Mode
 @param clearDiagnosticlMode clear Diagnosticl Mode
 @param clearNocturnalMode clear Nocturnal Mode
 */
- (void)clearHistoryDataWithSelectedUsualMode:(BOOL)clearUsualMode DiagnosticMode:(BOOL)clearDiagnosticlMode NocturnalMode:(BOOL)clearNocturnalMode;

/**
 Clear current mode history data of the BPM
 */
- (void)clearCurrentModeHistoryData;

/**
 Disconnect the Bluetooth with BPM
 */
- (void)disconnect;

/**
 Write device Time to BPM
 */
- (void)writeDeviceTime;

/**
 Write a new user ID to BPM
 
 @param ID user ID
 */
- (void)writeUserID:(NSString *)ID;

/**
 Read nocturnal mode setting
 */
- (void)readNocturnalModeSetting;

/**
 Change nocturnal mode setting 
 Note : This command requires matching hardware to set. You can use "readDeviceIDAndInfo(MicroLifeDeviceInfo.openNocturnalMode)" to check if the device supports it.

 @param open Nocturnal ON/OFF
 @param year year
 @param month month
 @param day day
 @param hour hour
 */
- (void)changeNocturnalModeSettingOn:(BOOL)open StartYear:(NSInteger)year StartMonth:(NSInteger)month StartDay:(NSInteger)day StartHour:(NSInteger)hour;

/**
 Read device ID and info from BPM
 */
- (void)readDeviceIDAndInfo;

/**
 Read device Time from BPM
 */
- (void)readDeviceTime;

/**
 Read user ID and version data from BPM
 */
- (void)readUserAndVersionData;

/**
 Read Nocturnal mode history data from BPM
 */
- (void)readNocturnalModeHistoryData;

#pragma mark - Feature
/**
 Get the MAC address of the currently bound scale
 
 @return Mac address, if no device is bound, return nil
 */
- (NSData *)getBindingDevice;

/**
 Binding device
 
 @param macAddress Target device's Mac address
 */
- (void)bindingDevice:(NSData *)macAddress;

/**
   Untie equipment
   */
- (void)unBindingDevice;

#pragma mark - Bluetooth connection
/**
 Re-search for device
 
 */
- (void)reScan;

/**
 Stop searching for device
 */
- (void)stopScan;

/**
   Disconnect Bluetooth from the current device
   */
- (void)disconnectDevice;
@end

NS_ASSUME_NONNULL_END
