//
//  ScaleBleManager.h
//  BodyFat-B-Demo
//
//  Created by 倍泰 on 2017/11/8.
//  Copyright © 2017年 ebelter. All rights reserved.
//
// 体脂秤蓝牙管理器
// 2018.01.11  版本V1.0.2  1.增加APP退入后台，回到前台的处理；2.增加手机蓝牙状态；3.更新体成分算法
// 2018.03.26  版本V1.0.3  1.修改运动员模式的计算系数
// 2018.11.28  版本V1.0.4  1.去除自动扫描功能，开放以下开始扫描和停止扫描两个接口
// 2019.04.23  版本V1.0.5  1.新增对心率体脂秤的支持
// 2019.06.28  版本V1.0.6  1.新增对 Body Fat-B2体脂秤 的支持
// 2019.07.02  版本V1.0.7  1.新增对GOQii体脂秤的支持
// 2019.07.10  版本V1.0.8  1.新增收到体重数据回调
// 2019.07.25  版本V1.0.9  1.扫描到设备，回调接口增加deviceName字段；2.设备蓝牙名改为只匹配“GOQii”

#import <Foundation/Foundation.h>
#import <UIKit/UIKit.h>
#import "BodyFat.h"
#import "OfflineBodyFat.h"
#import "UserInfo.h"

@class ScaleBleManager;

/*
 手机蓝牙状态
 */
typedef enum : NSInteger {
    SBMPhoneBLEStateUnknown = 0,
    SBMPhoneBLEStateResetting,      // The connection with the system service was momentarily lost, update imminent.
    SBMPhoneBLEStateUnsupported,    // The platform doesn't support the Bluetooth Low Energy Central/Client role.
    SBMPhoneBLEStateUnauthorized,   // The application is not authorized to use the Bluetooth Low Energy role.
    SBMPhoneBLEStatePoweredOff,     // Bluetooth is currently powered off.
    SBMPhoneBLEStatePoweredOn,      // Bluetooth is currently powered on and available to use.
} SBMPhoneBLEState;

typedef enum : int {
    BSDeviceTypeNormal,     // 普通体脂秤
    BSDeviceTypeHeart,      // 心率体脂秤（暂不支持离线测量）
} BSDeviceType;             // 连接的设备类型

@protocol ScaleBleManagerDelegate <NSObject>
#pragma mark - 蓝牙连接部分
@optional
/**
 *  手机蓝牙状态改变
 */
- (void)scaleBleManager:(ScaleBleManager *)manager cellPhoneBluetoothDidUpdateState:(SBMPhoneBLEState)state;

/**
 搜索到设备
 */
- (void)scaleBleManager:(ScaleBleManager *)manager didDiscoverBluetoothDeviceMacAddress:(NSData *)macAddress deviceName:(NSString *)deviceName;

/**
 设备已连接

 @param deviceType 设备类型：BSDeviceTypeNormal：普通体脂秤/BSDeviceTypeHeart：心率体脂秤
 */
- (void)scaleBleManagerDidConnectDevice:(ScaleBleManager *)manager deviceType:(BSDeviceType)deviceType;

/**
 设备连接失败
 */
- (void)scaleBleManagerDidFailToConnectDevice:(ScaleBleManager *)manager;


/**
 断开连接
 */
- (void)scaleBleManagerDidDisconnectDevice:(ScaleBleManager *)manager;

#pragma mark - 收到数据部分
/**
 收到秤被唤醒（秤面亮屏）
 */
- (void)scaleBleManagerDidReceiveScaleAwake:(ScaleBleManager *)manager;

/**
 收到秤进入休眠（秤面息屏）
 */
- (void)scaleBleManagerDidReceiveScaleSleep:(ScaleBleManager *)manager;

/**
 收到用户信息更新响应
 注：SDK内部也会下发用户信息给秤，所以您可能会频繁收到此回调消息
 */
- (void)scaleBleManagerDidReceiveUpdateUserInfoResponse:(ScaleBleManager *)manager;

/**
 收到当前测量的结果数据
 */
- (void)scaleBleManager:(ScaleBleManager *)manager didReceiveMeasureResult:(BodyFat *)bodyFat;

/**
 收到当前测量的体重（会在收到完整测量结果前收到）
 */
- (void)scaleBleManager:(ScaleBleManager *)manager didReceiveMeasureWeight:(float)weight unit:(EBUnitType)unit;

/**
 收到离线数据
 */
- (void)scaleBleManager:(ScaleBleManager *)manager didReceiveOfflineData:(OfflineBodyFat *)bodyFat;

/**
 收到无更多离线数据的指令
 */
- (void)scaleBleManagerDidReceiveNoMoreOfflineData:(ScaleBleManager *)manager;


/**
 收到清空所有用户响应
 */
- (void)scaleBleManagerDidReceiveClearDataResp:(ScaleBleManager *)manager;
@end

@interface ScaleBleManager : NSObject
@property (nonatomic,weak) id<ScaleBleManagerDelegate> delegate;
/**
 手机蓝牙状态
 */
@property (nonatomic,assign) SBMPhoneBLEState bleState;

/**
 设备连接状态
 */
@property (nonatomic,assign,readonly) BOOL isConnected;

+ (instancetype)shareInstance;
#pragma mark 下发

/**
 下发用户信息

 @param userInfo 用户信息model
 */
- (void)sendUserInfo:(UserInfo *)userInfo;

/**
 * 下发清除所有用户
 */
- (void)sendToClearAllUsers;

/**
 下发查询离线记录
 */
- (void)sendToRequestOfflineDataWithUserInfo:(UserInfo *)userInfo;

#pragma mark 功能性

/**
 获取当前已绑定的秤的Mac地址

 @return Mac地址，如没有绑定设备，返回nil
 */
- (NSData *)getBindingDevice;

/**
 绑定设备

 @param macAddress 目标设备的Mac地址
 */
- (void)bindingDevice:(NSData *)macAddress;

/**
 解绑设备
 */
- (void)unBindingDevice;

#pragma mark 蓝牙连接

/**
 开始扫描
 */
- (void)startScan;

/**
 停止扫描
 */
- (void)stopScan;

/**
 断开与当前设备的蓝牙连接
 */
- (void)disconnectDevice;

@end
